function genfirtbvhdl(dir, nametb, namefir, namefirref, fnstimin, fnstimout, filtspec)
%genfirtbvhdl(dir, nametb, namefir, namefirref, fnstimin, fnstimout, filtspec)
%
%Generates a testbench for an FIR filter:
%  name_tb.vhdl - testbench for verifying the function of the filter.
%    The testbench expects to find files namefir.vhdl and 
%    namefirref.vhdl, generated by genfirvhdl and genfirrefvhdl. The
%    files name_in.stim and name_out.stim are also expected to be found
%    in the same directory. These files contain test stimuli and the 
%    expected output of the filter. The files can be generated with
%    genfirstim.
%Inputs:
%  dir - the directory to generate the code in
%  nametb - the testbench entity name
%  namefir - the DUT FIR filter entity name
%  namefirref - the DUT FIR reference filter entity name
%  fnstimin - name of file containing input stimuli
%  fnstimout - name of file containing expected filter output
%  filtspec - filter specification, used field:
%    numphases: number of phases
%    wdata: data wordlength
%    wout: output wordlength

%Copyright (C) 2008 Anton Blad, Oscar Gustafsson.
%This file is licensed under a modified version of GPL v2, see the file
%LICENSE for details.

tbfile = sprintf('%s/%s.vhdl', dir, nametb);

fid = fopen(tbfile, 'w+');

fprintf(fid, 'library ieee;\n');
fprintf(fid, 'use ieee.std_logic_1164.all;\n');
fprintf(fid, 'use std.textio.all;\n');
fprintf(fid, 'use ieee.std_logic_textio.all;\n');
fprintf(fid, '\n');
fprintf(fid, 'entity %s is\n', nametb);
fprintf(fid, '\n');
fprintf(fid, 'end %s;\n', nametb);
fprintf(fid, '\n');
fprintf(fid, 'architecture generated of %s is\n', nametb);
fprintf(fid, '\n');
fprintf(fid, '  constant NumPhases : positive := %d;\n', filtspec.numphases);
fprintf(fid, '  constant WIn : positive := %d;\n', filtspec.wdata);
fprintf(fid, '  constant WOut : positive := %d;\n', filtspec.wout);
fprintf(fid, '\n');
fprintf(fid, '  constant infile : string := "%s";\n', fnstimin);
fprintf(fid, '  constant outfile : string := "%s";\n', fnstimout);
fprintf(fid, '\n');
fprintf(fid, '  constant period : time := 20 ns;\n');
fprintf(fid, '\n');
fprintf(fid, '  type x_array is array (0 to NumPhases-1) of std_logic_vector(WIn-1 downto 0);\n');
fprintf(fid, '\n');
fprintf(fid, '  component %s\n', namefir);
fprintf(fid, '    port (\n');
fprintf(fid, '      clk, reset : in  std_logic;\n');
for phase = 0:filtspec.numphases-1
	fprintf(fid, '      x_%d        : in  std_logic_vector(WIn-1 downto 0);\n', phase);
end
fprintf(fid, '      y          : out std_logic_vector(WOut-1 downto 0));\n');
fprintf(fid, '  end component;\n');
fprintf(fid, '\n');
fprintf(fid, '  component %s\n', namefirref);
fprintf(fid, '    port (\n');
fprintf(fid, '      clk, reset : in  std_logic;\n');
for phase = 0:filtspec.numphases-1
	fprintf(fid, '      x_%d        : in  std_logic_vector(WIn-1 downto 0);\n', phase);
end
fprintf(fid, '      y          : out std_logic_vector(WOut-1 downto 0));\n');
fprintf(fid, '  end component;\n');
fprintf(fid, '\n');
fprintf(fid, '  signal clk : std_logic := ''0'';\n');
fprintf(fid, '  signal reset : std_logic := ''1'';\n');
fprintf(fid, '\n');
fprintf(fid, '  signal x : x_array;\n');
fprintf(fid, '  signal y : std_logic_vector(WOut-1 downto 0);\n');
fprintf(fid, '  signal yref : std_logic_vector(WOut-1 downto 0);\n');
fprintf(fid, '  signal ytrue : std_logic_vector(WOut-1 downto 0);\n');
fprintf(fid, '  signal check : std_logic;\n');
fprintf(fid, '  signal checkref : std_logic;\n');
fprintf(fid, '\n');
fprintf(fid, 'begin  -- test\n');
fprintf(fid, '\n');
fprintf(fid, '  reset <= ''0'', ''1'' after period;\n');
fprintf(fid, '  clk <= not clk after period/2;\n');
fprintf(fid, '\n');
fprintf(fid, '  tester: process\n');
fprintf(fid, '    variable l : line;\n');
fprintf(fid, '    file datain : text open read_mode is infile;\n');
fprintf(fid, '    file dataout : text open read_mode is outfile;\n');
fprintf(fid, '\n');
fprintf(fid, '    variable x_v : x_array;\n');
fprintf(fid, '    variable y_v : std_logic_vector(WOut-1 downto 0);\n');
fprintf(fid, '\n');
fprintf(fid, '  begin\n');
fprintf(fid, '\n');
fprintf(fid, '    wait until reset = ''1'';\n');
fprintf(fid, '\n');
fprintf(fid, '    loop\n');
fprintf(fid, '      readline(datain, l);\n');
fprintf(fid, '      for i in 0 to NumPhases-1 loop\n');
fprintf(fid, '        read(l, x_v(i));\n');
fprintf(fid, '      end loop;  -- i\n');
fprintf(fid, '      x <= x_v;\n');
fprintf(fid, '\n');
fprintf(fid, '      readline(dataout, l);\n');
fprintf(fid, '      read(l, y_v);\n');
fprintf(fid, '      ytrue <= y_v;\n');
fprintf(fid, '\n');
fprintf(fid, '      wait until rising_edge(clk);\n');
fprintf(fid, '\n');
fprintf(fid, '    end loop;\n');
fprintf(fid, '\n');
fprintf(fid, '  end process tester;\n');
fprintf(fid, '\n');
fprintf(fid, '  DUT: %s\n', namefir);
fprintf(fid, '    port map (\n');
fprintf(fid, '      clk   => clk,\n');
fprintf(fid, '      reset => reset,\n');
for phase = 0:filtspec.numphases-1
	fprintf(fid, '      x_%d   => x(%d),\n', phase, phase);
end
fprintf(fid, '      y     => y);\n');
fprintf(fid, '\n');
fprintf(fid, '  DUTref: %s\n', namefirref);
fprintf(fid, '    port map (\n');
fprintf(fid, '      clk   => clk,\n');
fprintf(fid, '      reset => reset,\n');
for phase = 0:filtspec.numphases-1
	fprintf(fid, '      x_%d   => x(%d),\n', phase, phase);
end
fprintf(fid, '      y     => yref);\n');
fprintf(fid, '\n');
fprintf(fid, '  process(clk, reset)\n');
fprintf(fid, '  begin\n');
fprintf(fid, '    if reset = ''0'' then\n');
fprintf(fid, '      check <= ''0'';\n');
fprintf(fid, '    elsif clk''event and clk = ''1'' then\n');
fprintf(fid, '      if y = ytrue then\n');
fprintf(fid, '        check <= ''0'';\n');
fprintf(fid, '      else\n');
fprintf(fid, '        check <= ''1'';\n');
fprintf(fid, '      end if;\n');
fprintf(fid, '    end if;\n');
fprintf(fid, '  end process;\n');
fprintf(fid, '\n');
fprintf(fid, '  process(clk, reset)\n');
fprintf(fid, '  begin\n');
fprintf(fid, '    if reset = ''0'' then\n');
fprintf(fid, '      checkref <= ''0'';\n');
fprintf(fid, '    elsif clk''event and clk = ''1'' then\n');
fprintf(fid, '      if yref = ytrue then\n');
fprintf(fid, '        checkref <= ''0'';\n');
fprintf(fid, '      else\n');
fprintf(fid, '        checkref <= ''1'';\n');
fprintf(fid, '      end if;\n');
fprintf(fid, '    end if;\n');
fprintf(fid, '  end process;\n');
fprintf(fid, '\n');
fprintf(fid, 'end generated;\n');

fclose(fid);

